//
//  APObject.m
//  APlayer
//
//  Created by Holger Sadewasser on 4/25/08.
//  Copyright 2008. All rights reserved.
//

#import "parameter.h"
#import "APObject.h"
#import "tools.h"


//#define RADUIS_ASTEROID_LARGE  35.2
//#define RADUIS_ASTEROID_MEDIUM 19.2
//#define RADUIS_ASTEROID_SMALL  10.4
//#define RADUIS_SHIP            15.2

#define RADUIS_ASTEROID_LARGE  40.0
#define RADUIS_ASTEROID_MEDIUM 20.0
#define RADUIS_ASTEROID_SMALL  8.0
#define RADUIS_SHIP            30.0

#define TIME_TO_LIVE_EXPLOSION 37


extern BOOL gFlgLogging;


typedef struct APCorrection {
  int    index;
  double correction;
} APCorrection_t;

static APCorrection_t gPackedCorrections[44] = {
  {0x00, 0.0  },
  
  {0x01, 0.0  },
  {0x02, 0.125},
  {0x04, 0.25 },
  {0x08, 0.375},
  {0x10, 0.5  },
  {0x20, 0.625},
  {0x40, 0.75 },
  {0x80, 0.875},

  {0x11, 0.0  },
  {0x22, 0.25 },
  {0x44, 0.5  },
  {0x88, 0.75 },
  
  {0x25, 0.0  },
  {0x29, 0.125},
  {0x49, 0.25 },
  {0x4a, 0.375},
  {0x52, 0.5  },
  {0x92, 0.625},
  {0x94, 0.75 },
  {0xa4, 0.875},

  {0x55, 0.25 },
  {0xaa, 0.75 },

  {0x5b, 0.0  },
  {0x6b, 0.125},
  {0x6d, 0.25 },
  {0xad, 0.375},
  {0xb5, 0.5  },
  {0xb6, 0.625},
  {0xd6, 0.75 },
  {0xda, 0.875},

  {0x77, 0.0  },
  {0xbb, 0.25 },
  {0xdd, 0.5  },
  {0xee, 0.75 },
  
  {0x7f, 0.0  },
  {0xbf, 0.125},
  {0xdf, 0.25 },
  {0xef, 0.375},
  {0xf7, 0.5  },
  {0xfb, 0.625},
  {0xfd, 0.75 },
  {0xfe, 0.875},
  
  {0xff, 0.5  },
};

static double gCorrections[256];

// Next object ID
static int gNextID = 0;


// methods that are used internally by this class, but should not be needed externally
// are defined here to keep them "private". Anyone who knows about them can still call
// them of course, but by putting the prototypes here, it emphasizes that they are only
// meant for internal use.
@interface APObject(_private_methods)

-(double)_subpixelCorrectionForOffset:(int)iOffset positions:(int *)iPositions distFunction:(int(*)(int,int))iDistFunction distance:(int)iDistance;

@end


@implementation APObject

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Class methods
// -------------------------------------------------------------------------------------

+(void)initialize
{
  static BOOL initialized = NO;
  int i;
  
  if (!initialized) {
    bzero( &gCorrections, sizeof(gCorrections) );
    for ( i = 0; i < sizeof(gPackedCorrections) / sizeof(APCorrection_t); i++ ) {
      gCorrections[gPackedCorrections[i].index] = gPackedCorrections[i].correction;
    }
    initialized = YES;
  }
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Initializers/Clean up
// -------------------------------------------------------------------------------------

-(id)initWithType:(int)iType size:(int)iSize posX:(int)iPosX posY:(int)iPosY {
  
  self = [super init];
  if (self != nil) {
    mID = gNextID++;
//    NSLog(@"%p  ID: %d  type: %d", self, mID, iType);
    mType = iType;
    mSize = iSize;
    
    mInitPosX = iPosX;
    mInitPosY = iPosY;
    mTempPosX = mInitPosX;
    mTempPosY = mInitPosY;
    mDeltaX[0] = iPosX;
    mDeltaY[0] = iPosY;
    mCorrX = 0.0;
    mCorrY = 0.0;
    
//    mHeadingX = 0.0;
//    mHeadingY = 0.0;
    mFlgCalcHeading = YES;
    
    if ( mType == cShip ) {
      mRadius = RADUIS_SHIP;      
    } else if ( mType != cShot && mType != cEmpty ) {
      switch ( mSize ) {
        case 0:
          mRadius = RADUIS_ASTEROID_LARGE;
          break;
        case 15:
          mRadius = RADUIS_ASTEROID_MEDIUM;
          break;
        case 14:
          mRadius = RADUIS_ASTEROID_SMALL;
          break;
      }
    } else {
      mRadius = 0.0;
    }
    mRadius2 = mRadius*mRadius;
    
    mFlgHyperspace = (mType != cShot);

    // Before calling this method mType and mSize must have been set.
    [self _setPosX:iPosX posY:iPosY];

    mInitFrame = 1;
    mTempFrame = mInitFrame;
    mTotalFrames = mInitFrame;

    if ( iType == cShot )
      mTimeToLive = 72;
    else
      mTimeToLive = -1;
    
    mNextAdaption = mTotalFrames + AVERAGING_INTERVAL;
    
//- Begin Records shot angles ---------------------
//    mFlgRecording = NO;
//    mShipViewX = 0;
//    mShipViewY = 0;
//- End Records shot angles ---------------------
    
  }
  return self;
}

-(id)initWithType:(int)iType posX:(int)iPosX posY:(int)iPosY {
  return [self initWithType:iType size:0 posX:iPosX posY:iPosY];
}

-(id)initWithType:(int)iType {
  return [self initWithType:iType size:0 posX:0 posY:0];
}

-(id)init {
  return [self initWithType:cEmpty size:0 posX:0 posY:0];
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Instance Methods
// -------------------------------------------------------------------------------------

-(BOOL)matchesObjectWithType:(int)iType size:(int)iSize posX:(int)iPosX posY:(int)iPosY atDistance:(double *)eDist {
  
  double distX, distY, dist;
  double distXa, distYa;
  double divX, divY, div;
  
  if ( mType == iType && mSize == iSize ) {
    distX = distXdouble( mPosX, (double)iPosX );
    distY = distYdouble( mPosY, (double)iPosY );      
    distXa = fabs(distX);
    distYa = fabs(distY);
    dist = distX*distX + distY*distY;
    //      if ( dist < TRACKING_POS_RANGE  )
//    if ( (iType == cShot && distXa < TRACKING_POS_RANGE+1.0 && distYa < TRACKING_POS_RANGE+1.0) ||
//         (distXa < TRACKING_POS_RANGE && distYa < TRACKING_POS_RANGE) ) {
    if ( distXa < TRACKING_POS_RANGE && distYa < TRACKING_POS_RANGE ) {
      if ( mTotalFrames == 1 ) {
        *eDist = dist;
        return YES;
      } else {
        divX = fabs(distX - mHeadingX);
        divY = fabs(distY - mHeadingY);
        div = divX*divX + divY*divY;
        //          if ( div < TRACKING_HEADING_DIV)
        if ( divX < TRACKING_HEADING_DIV && divY < TRACKING_HEADING_DIV ) {
          *eDist = div;
          return YES;
        }
      }
    }
  }  
  return NO;
}

-(void)updateWithPosX:(int)iPosX posY:(int)iPosY frames:(uint8_t)iFrames {
  
  int distX, distY;
  
  mTotalFrames += iFrames;
//  if ( mType != cShot && mTimeToLive >= 0 && mTimeToLive < iFrames )
//    NSLog(@"Missed object at %d %d", iPosX, iPosY);
  mTimeToLive  -= iFrames;
  
  if ( mTotalFrames > 2 )
    mFlgHyperspace = YES;
  
//  if ( mTotalFrames >= mNextAdaption )
//  {
////- Begin Records shot angles ---------------------
////    if ( mType == cShot && mFlgRecording == YES ) {
////      mFlgRecording = NO;
////      double dx = ((double)iPosX-mInitPosX)/4.0;
////      double dy = ((double)iPosY-mInitPosY)/4.0;
////      double a;
////      if ( dy < 0.0 )
////        a = 2.0 * M_PI - acos(dx/sqrt(dx*dx+dy*dy));
////      else        
////        a = acos(dx/sqrt(dx*dx+dy*dy));
////      NSLog(@"shot p: %3d\tab: %3d\tvx: %5d\t vy: %5d\tx: %6.2lf\ty: %6.2lf\ta: %lf\t b: %lf", (int)mPing, (int)mAngleByteTmp, (int)mShipViewX, (int)mShipViewY, dx, dy, a, 360.0*a/(2.0*M_PI));
////    }
////- End Records shot angles ---------------------
//
//    mInitPosX = mTempPosX;
//    mInitPosY = mTempPosY;
//    mInitFrame = mTempFrame;
//    
//    mTempPosX = iPosX;
//    mTempPosY = iPosY;
//    mTempFrame = mTotalFrames;
//    
//    mNextAdaption = mTotalFrames + AVERAGING_INTERVAL;
//  }
  
  [self _setPosX:iPosX posY:iPosY];
  
  uint8_t deltaFrames = mTotalFrames - mInitFrame;
  
  distX = distXint( (int)mInitPosX, iPosX );
  distY = distYint( (int)mInitPosY, iPosY );

  if ( deltaFrames <= 8 ) {
    mDeltaX[deltaFrames] = iPosX;
    mDeltaY[deltaFrames] = iPosY;
  }
  if ( (deltaFrames & 0x07) == 0 ) {
    mHeadingX = (double)distX / (double)deltaFrames;
    mHeadingY = (double)distY / (double)deltaFrames;
    
    mFlgCalcHeading = NO;
    
    mCorrX = [self _subpixelCorrectionForOffset:(int)mHeadingX positions:mDeltaX distFunction:distXint distance:distX];
    mCorrY = [self _subpixelCorrectionForOffset:(int)mHeadingY positions:mDeltaY distFunction:distYint distance:distY];
    
    mInitPosX = iPosX;
    mInitPosY = iPosY;
    mInitFrame = mTotalFrames;
    mDeltaX[0] = iPosX;
    mDeltaY[0] = iPosY;
    //      gFlgLogging = YES;
  } else if ( mFlgCalcHeading == YES ) {
    mHeadingX = (double)distX / (double)deltaFrames;
    mHeadingY = (double)distY / (double)deltaFrames;
  }
  
  //    if ( mID == 3 ) {
  //      NSLog(@"hx: %f  hy: %f  frames: %d", mHeadingX, mHeadingY, mTotalFrames-mInitFrame);
  //    }
  //  } else {
  //    mPosX += iFrames * mHeadingX;
  //    mPosY += iFrames * mHeadingY;
  
  mPosX += mCorrX;
  mPosY += mCorrY;
  
}


-(double)timeOfCollisionWithPosX:(double)a0 posY:(double)b0 dirX:(double)iDirX dirY:(double)iDirY radius:(double)iRadius delay:(int)iDelay accuracy:(double *)xAccuracy {
         
  static int count = -1;
  
  double a = iDirX;
  double b = iDirY;
  
  double x0 = a0 + distXdouble( a0, mPosX );
  double y0 = b0 + distYdouble( b0, mPosY );

//  double x0 = mPosX;
//  double y0 = mPosY;

  double x = mHeadingX;
  double y = mHeadingY;
  double xn, yn;
  
  double p, q, n, diff, diff2;
  
  double c, d, angle, lxy, lxy2;
  double dist, lab, lab2, labxy;
  double ca;
  
  *xAccuracy = 1e10;
  
  if ( a == 0.0 && b == 0.0 || x == 0.0 && y == 0.0 ) return -1.0;
  
  lab2 = a*a + b*b;
  lab = sqrt(lab2);
  lxy2 = x*x + y*y;
  lxy = sqrt(lxy2);
  labxy = lab*lxy;
  
  // Check if the two vectors are (almost) parallel.
  ca = (a*x+b*y)/labxy;

  
//  s = n / labxy;
//  NSLog(@"n: %f  ca: %f", n, ca);
//  if ( fabs(s) < 0.05 ) {
  
  if ( fabs(ca) > 0.98 ) {
    
//    dx = x0 - a0;
//    dy = y0 - b0;
//    dist = sqrt(dx*dx + dy*dy);
//    diff = dist/tan(asin(s));
    
    // Check if the other object is heading towards us
    if ( (x0-a0)*a + (y0-b0)*b < 0.0 ) return -1.0;

    // turn the direction by 90 degrees
    xn = -y;
    yn = x;
    
    n = a*yn - b*xn;
    //    p = (x0*yn + b0*xn - a0*yn - y0*xn) / n;
    p = ((x0-a0)*yn + (b0-y0)*xn) / n;
    if ( p > 0.0 ) {
      c = a0 + p*a - x0;
      d = b0 + p*b - y0;
      diff2 = c*c + d*d;
      if ( diff2 < mRadius2 ) {
        dist = p*lab - mRadius;
        if ( ca < 0.0 )
          n = lab + lxy;
        else
          n = lab - lxy;
        if ( n != 0.0 ) {
          if ( n < 0.0 )
            NSLog(@"n negative: %f", n);
          *xAccuracy = diff2;
          return dist / n;
        }
      }
    }
//    return -1.0;
  }

  
  n = a*y - b*x;
  if ( n == 0.0 ) return -1.0;
//  p = (x0*y + b0*x - a0*y - y0*x) / n;
  p = ((x0-a0)*y + (b0-y0)*x) / n;
  
  
  if ( gFlgLogging == YES ) {
//    if ( count < 256 )
//      NSLog(@"p = %f", p);
  }

  // Check if the time of intersection lies within the living time of a shot
  if ( p > 0.0 /*&& p < 72.0*/ ) {
//    q = (x0*b + b0*a - a0*b - y0*a) / n;
    q = ((x0-a0)*b + (b0-y0)*a) / n;
    
#ifdef INTERSECTION_TEST
    mQ = q;
#endif

    if ( gFlgLogging == YES ) {
//      if ( count < 256 )
//        NSLog(@"q = %f", q);
    }
    
    // Check if the object is near the point of intersection when the shot arrives there
    if ( q > 0.0 ) {
      angle = acos(ca);
      if ( angle > M_PI_2 )
        angle = M_PI - angle;
      d = mRadius / tan(angle);
      
//      if ( angle > M_PI_2 )
//        angle = angle - M_PI_2;
//      else
//        angle = M_PI_2 - angle;
//      d = tan(mRadius * angle);

      c = d*d + mRadius2;
      diff = p - q + (double)iDelay;
      diff2 = diff * diff;
      
#ifdef INTERSECTION_TEST
      mC = c;
#endif
      if ( gFlgLogging == YES ) {
        //    if ( ++count < 256 )
        NSLog(@"x0 = %f  y0 = %f  x = %f  y = %f  a0 = %f  b0 = %f  a = %f  b = %f  r: %f  delay: %4d  c: %f  d: %f  p: %f  q: %f", x0, y0, x, y, a0, b0, a, b, iRadius, iDelay, c, d, p, q);
      }
      
      if ( diff2*lxy2 < c ) {
//      if ( diff < mRadius2/(x*x+y*y))
#ifdef INTERSECTION_TEST
        NSLog(@"a: %f  b: %f  q: %f  c: %f  diff2: %f  lxy2: %f", a, b, q, c, diff, lxy2);
#endif
        *xAccuracy = diff2;
        return p;
      }
    }
  }
  return -1.0;
}


//-(double)timeOfCollisionWithPosX:(double)a0 posY:(double)b0 dirX:(double)a dirY:(double)b radius:(double)iRadius delay:(int)iDelay {
//  
//  static int count = -1;
//  
//  double c0 = 0.0;
//  
//  double c = 1.0;
//  
//  double x0 = mPosX;
//  double y0 = mPosY;
//  double z0 = -iDelay;
//  
//  double x = mHeadingX;
//  double y = mHeadingY;
//  double z = 1.0;
//  
//  double dxa, dyb, dzc;
//  double uu, vv, uv, ud, vd;
//  double p, q, n;
//  
//  //  if ( mID == 3 ) {
//  //    if ( ++count < 256 )
//  //      NSLog(@"x0 = %f  y0 = %f  z0 = %f  x = %f  y = %f  a0 = %f  b0 = %f  a = %f  b = %f", x0, y0, z0, x, y, a0, b0, a, b);
//  //  }
//  
//  // Check if the two vectors are parallel. If yes we assume there won't be a collision.
//  if ( b*z - c*y == 0.0 && c*x - a*z == 0.0 && a*y - b*x == 0.0 ) {
//    if ( mID == 3 ) {
//      NSLog(@"Vectors are parallel");
//    }
//    return -1.0;
//  }
//  
//  dxa = x0 - a0;
//  dyb = y0 - b0;
//  dzc = z0 - c0;
//  
//  uu = a*a + b*b + c*c;
//  vv = x*x + y*y + z*z;
//  uv = a*x + b*y + c*z;
//  ud = a*dxa + b*dyb + c*dzc;
//  vd = -(x*dxa + y*dyb + z*dzc);
//  n = uu*vv + uv*uv;
//  
//  if ( n == 0.0 ) return -1.0;
//  
//  p = (vv*ud + uv*vd) / n;
//  
//  //  if ( mID == 3 ) {
//  //    if ( count < 256 )
//  //      NSLog(@"p = %f", p);
//  //  }
//  
//  if ( p < 72.0 ) {
//    q = (uu*vd - uv*ud) / n;
//    
//    double dx = dxa + q*x - p*a;
//    double dy = dyb + q*y - p*b;
//    double dz = dzc + q*z - p*c;
//    double dist = mRadius + iRadius;
//    
//    //    if ( mID == 3 ) {
//    //      if ( count < 256 )
//    //        NSLog(@"dist: %f -> %f  calc dist: %f", dist, dist*dist, dx*dx + dy*dy + dz*dz);
//    //    }
//    
//    if ( dx*dx + dy*dy + dz*dz < dist*dist )
//      return p;
//  }
//  return -1.0;
//}


-(void)setTimeToLive:(int)iTimeToLive {
  mTimeToLive = iTimeToLive;
}

-(int)timeToLive {
  return mTimeToLive;
}

-(int)ID {
  return mID;
}

-(APObjectTypes_t)type {
  return mType;
}

-(int)size {
  return mSize;
}

-(double)posX {
  return mPosX;
}

-(double)posY {
  return mPosY;
}

-(int)posXInt {
  return mPosXInt;
}

-(int)posYInt {
  return mPosYInt;  
}

-(double)headingX {
  return mHeadingX;
}

-(double)headingY {
  return mHeadingY;
}

-(double)radius {
  return mRadius;
}

-(unsigned int)frames {
  return mTotalFrames;
}

-(BOOL)isAtPosX:(int)iPosX posY:(int)iPosY {
  return ( iPosX == mPosXInt && iPosY == mPosYInt );
}

-(void)makeExplosion {
  mType = cExplosion;
  [self setTimeToLive:TIME_TO_LIVE_EXPLOSION+1];
}

-(void)decreaseTimeToLiveBy:(NSNumber *)iFrames {
  mTimeToLive -= [iFrames intValue];
  if ( mType == cExplosion && mTimeToLive <= 0 ) [self init];
}


-(void)_setPosX:(int)iPosX posY:(int)iPosY {
  
  mPosX = (double)iPosX;
  mPosY = (double)iPosY;
  mPosXInt = iPosX;
  mPosYInt = iPosY;
  
  mBBx1 = mPosX - mRadius;
  mBBx2 = mPosX + mRadius;
  mBBy1 = mPosY - mRadius;
  mBBy2 = mPosY + mRadius;
  
}

#if defined WITH_WINDOW || defined INTERSECTION_TEST
-(void)draw {
  if ( mRadius == 0.0 )
    NSFrameRect(NSMakeRect( (float)(mPosX-2.0), (float)(mPosY-2.0), (float)5.0, (float)5.0 ));
  else 
    NSFrameRect(NSMakeRect( (float)mBBx1, (float)mBBy1, (float)(mBBx2-mBBx1), (float)(mBBy2-mBBy1) ));
  
  [NSBezierPath strokeLineFromPoint:NSMakePoint((float)mPosX,(float)mPosY) toPoint:NSMakePoint((float)(mPosX+150.0*mHeadingX),(float)(mPosY+150.0*mHeadingY))];
}
#endif

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Private Methods
// -------------------------------------------------------------------------------------

-(double)_subpixelCorrectionForOffset:(int)iOffset positions:(int *)iPositions distFunction:(int(*)(int,int))iDistFunction distance:(int)iDistance {

  int i;
  int index = 0;
  
  for ( i = 1; i < 9; i++ ) {
    index <<= 1;
    if ( iDistFunction( iPositions[i-1], iPositions[i] ) - iOffset != 0 ) index |= 1;
  }
//  NSLog(@"id: %d p: %d %d %d %d %d %d %d %d %d d: %d o: %d i: %d corr: %f", mID, iPositions[0], iPositions[1], iPositions[2], iPositions[3], iPositions[4], iPositions[5], iPositions[6], iPositions[7], iPositions[8], iDistance, iOffset, index, gCorrections[index]);
  if ( iDistance < 0 )
    return 0.875 - gCorrections[index];
  else
    return gCorrections[index];
}


@end
